/*
 * SPDX-FileCopyrightText: 2011-2025 Univention GmbH
 * SPDX-License-Identifier: AGPL-3.0-only
 */
/*global define*/

define([
	"dojo/_base/declare",
	"dojo/_base/lang",
	"umc/tools",
	"umc/store",
	"umc/widgets/Page",
	"umc/widgets/Grid",
	"umc/widgets/SearchForm",
	"umc/widgets/SearchBox",
	"umc/widgets/ComboBox",
	"umc/i18n!umc/modules/printers"
], function(declare, lang, tools, store, Page, Grid, SearchForm, SearchBox, ComboBox, _) {
	return declare("umc.modules.printers.OverviewPage", [ Page ], {

		_last_filter: { key: 'printer', pattern: '' },

		postMixInProperties: function() {
			lang.mixin(this,{
				helpText: _("This module allows to manage print jobs of printers on your local machine."),
				headerText: _("Printer administration")
			});

			this.inherited(arguments);
		},

		fullWidth: true,

		buildRendering: function() {
			this.inherited(arguments);

			this._form = new SearchForm({
				region: 'nav',
				hideSubmitButton: true,
				widgets: [{
					name: 'key',
					'class': 'umcTextBoxOnBody',
					type: ComboBox,
					label: _("Search for"),
					staticValues: [
					 	 { id: 'printer',		label: _("Printer name")},
					 	 { id: 'description',	label: _("Description")},
					 	 { id: 'location',		label: _("Location") }
					],
					sortStaticValues: false
				}, {
					name: 'pattern',
					type: SearchBox,
					'class': 'umcTextBoxOnBody',
					inlineLabel: _('Search...'),
					value: '',
					onSearch: lang.hitch(this, function() {
						this._form.submit();
					})
				}],
				layout: [['key', 'pattern']],
				onSearch: lang.hitch(this, function(values) {
					this._enable_search_button(false);
					this._last_filter = values; // save for easy refresh
					this._grid.filter(values);
				})
			});
			this._enable_search_button(false);
			this.addChild(this._form);

			var columns = [{
				name: 'server',
				label: _("Server")
			}, {
				name: 'printer',
				label: _("Printer")
			}, {
				name: 'status',
				label: _("Status"),
				// 'enabled'/'disabled' are kind of keywords, just as they're returned
				// from cups if invoked without locale (LANG=C).
				// Our wording for this is 'active'/'inactive'.
				formatter: lang.hitch(this, function(value) {
					switch(value)
					{
						case 'enabled': return _("active");
						case 'disabled': return _("inactive");
					}
					return _("unknown");
				})
			}, {
				name: 'location',
				label: _("Location")
			}, {
				name: 'description',
				label: _("Description")
			}];

			var actions = [{
				name: 'open',
				label: _("View details"),
				isStandardAction: true,
				callback: lang.hitch(this,function(id, values) {
					// 2.4 uses the printer ID as key property, so we do that as well.
					this.openDetail(id[0]);
				})
			}, {
				name: 'activate',
				label: _("Activate"),
				isStandardAction: true,
				callback: lang.hitch(this, function(ids) {
					// no multi action for now, but who knows...
					for (var p in ids)
					{
						tools.umcpCommand('printers/enable', { printer: ids[p], on: true }).then(lang.hitch(this, function() {
							this._refresh_view();
						}));
					}
				}),
				canExecute: lang.hitch(this, function(values) {
					return (values.status == 'disabled');
				})
			}, {
				name: 'deactivate',
				label: _("Deactivate"),
				isStandardAction: true,
				callback: lang.hitch(this, function(ids) {
					// no multi action for now, but who knows...
					for (var p in ids)
					{
						tools.umcpCommand('printers/enable', { printer: ids[p], on: false }).then(lang.hitch(this, function() {
							this._refresh_view();
						}));
					}
				}),
				canExecute: lang.hitch(this, function(values) {
					return (values.status == 'enabled');
				})
			}, {
				name: 'refresh',
				label: _("Refresh printer list"),
				isContextAction: false,
				callback: lang.hitch(this, function() {
					this._refresh_view();
				})
			}];

			this._grid = new Grid({
				columns: columns,
				region: 'main',
				actions: actions,
				defaultAction: 'open',
				moduleStore: store('printer','printers'),
				// fill grid on first open
				query: {key:'printer', pattern: '*'},
				onFilterDone: lang.hitch(this, function(success) {
					this._enable_search_button(true);
				})
			});
			this.addChild(this._grid);
		},

		_enable_search_button: function(on) {
			this._form._buttons.submit.setDisabled(! on);
		},

		// refreshes the grid. can be called manually (pressing the refresh button)
		// or automatically (as response to the activate/deactivate action)
		_refresh_view: function() {
			this._grid.filter(this._last_filter);
		},

		// when we come back from any kind of detail view that
		// could have invoked some actions... refresh our view.
		onShow: function() {
			this._refresh_view();
		},

		// DetailPage gives results back here.
		setArgs: function(args) {
		},

		// main module listens here, to switch to the detail view.
		// args can propagate the id of the printer to show
		openDetail: function(args) {
		},
	});
});
